
/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential Information of Monotype Imaging Inc. */

/* fs_bitmap.c */


#include "fs_itype.h"

#ifdef FS_BITMAPS

#include "fs_bitmap.h"
#include "fs_effects.h"

/* these are used only in this file */
static FS_CONST FS_BYTE Lmask[8] = {0xFF, 0x7F, 0x3F, 0x1F, 0x0F, 0x07, 0x03, 0x01};
static FS_CONST FS_BYTE Rmask[8] = {0x80, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC, 0xFE, 0xFF};

/****************************************************************/
/* allocate a new bitmap for a scaled outline <p> */
FS_BITMAP *new_bitmap(_DS_ FS_OUTLINE *outl)
{
    FS_SHORT lo_x, hi_x, lo_y, hi_y;
    FS_SHORT width, bpl, height;
    FS_LONG n;
    FS_BITMAP *bmap;

    /* space characters */
    if (outl->num == 0)
    {
        n = offsetof(FS_BITMAP, bits);
#ifdef FS_MEM_DBG
        STATE.memdbgid = "FS_BITMAP";
#endif
        bmap = (FS_BITMAP *) FSS_calloc(_PS_ n);
        if (bmap == 0)
            return 0;
        bmap->size = n;
        bmap->i_dx = outl->i_dx;
        bmap->i_dy = outl->i_dy;
        bmap->dx = outl->dx;
        bmap->dy = outl->dy;
        bmap->type = FS_MAP_BITMAP;
        bmap->bitsPerPixel = 1;
        return bmap;
    }

#ifdef FS_STIK
    /* if stick character leave extra room to expand sticks */
    if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK) &&
        outl->nc && !(outl->type[0] & OUTLINE_CHAR))
    {
        SENV *senv = (SENV *)(STATE.cur_sfnt->senv);
        FS_FIXED w;

        w = senv->yppm * STATE.cur_sfnt->stroke_pct;

        w = (w + 0x00008000) & 0xFFFF0000;
        w >>= 1;
        w += FIXED_ONE;    /* add one extra pixel ... autohint can move the wrong way */

        lo_x = FS_FLOOR(outl->lo_x - w);
        hi_x = FS_CEIL(outl->hi_x + w);
        lo_y = FS_FLOOR(outl->lo_y - w);
        hi_y = FS_CEIL(outl->hi_y + w);
    }
    else
    {
        /* normal characters */
        lo_x = FS_FLOOR(outl->lo_x);
        hi_x = FS_CEIL(outl->hi_x);
        lo_y = FS_FLOOR(outl->lo_y);
        hi_y = FS_CEIL(outl->hi_y);
    }
#else
    /* normal characters */
    lo_x = FS_FLOOR(outl->lo_x);
    hi_x = FS_CEIL(outl->hi_x);
    lo_y = FS_FLOOR(outl->lo_y);
    hi_y = FS_CEIL(outl->hi_y);
#endif /* FS_STIK */

#ifdef FS_PSEUDO_BOLD
    if ( (STATE.flags & FLAGS_ADD_WEIGHT) ||
         ((STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON) &&
         (STATE.lpm <= 26) && !(STATE.flags & FLAGS_OUTLINEBOLD)) )
    {
        SENV *senv = (SENV *)(STATE.cur_sfnt->senv);
        FS_SHORT bw = senv->bold_width;
        if (bw)
            lo_x--;
        if (bw > 1)
            hi_x++;
    }
#endif

    width = hi_x - lo_x + 1;
    bpl = (7 + width) / 8;
    height = hi_y - lo_y + 1;

    /*** hmmm ***/
    if (width <= 0 || width >= 16384 || height <= 0 || height >= 16384)
    {
        return 0;
    }

    /* create it */
    n = offsetof(FS_BITMAP, bits);
    n += (FS_LONG)bpl * height;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_BITMAP";
#endif
    bmap = (FS_BITMAP *) FSS_calloc(_PS_ n);
    if (0 == bmap)
        return 0;

    /* set the fields */
    bmap->size = n;
    bmap->lo_x = lo_x;
    bmap->hi_y = hi_y;
    bmap->height = height;
    bmap->width = width;
    bmap->bpl = bpl;
    bmap->i_dx = outl->i_dx;
    bmap->i_dy = outl->i_dy;
    bmap->dx = outl->dx;
    bmap->dy = outl->dy;
    bmap->type = FS_MAP_BITMAP;
    bmap->bitsPerPixel = 1;
    return bmap;
}

/*************************************************************************/
/*lint -e661 Warning -- Possible access of out-of-bounds pointer         */
/*lint -e662 Warning -- Possible creation of out-of-bounds pointer       */
/*************************************************************************/
/* set NZW runs based on the transition array, w/ xDropout processing    */
FS_VOID nzw_runs(_DS_ FS_BITMAP *bitmap, TLIST *list)
{
    int index, i, winding;
    FS_FIXED x0, x1, t;
    int c0, c1, n, lo_x = bitmap->lo_x;
    FS_BYTE *bits;
    FS_BYTE *start_bits;
    int which[2] = { -1, 1};
    TNODE *p, *nodes = (TNODE *)(list->nodes);
    FS_ULONG fix = (STATE.flags & FLAGS_FIX_DROPOUTS);

    STATE.error = SUCCESS;
    start_bits = bitmap->bits;

    /* top of list not *always* top of bitmap */
    n = bitmap->hi_y - list->max;
    if (n)
        start_bits += n * bitmap->bpl;

    bits = start_bits;

    for (i = 0; i < list->num; i++)
    {
        index = list->indices[i];
        /* process a row of intercepts */
        while (index >= 0)
        {
            /* get next transition */
            p = nodes + index;
            index = p->next;
            if (index < 0)
            {
#ifndef EDIT_MODE
                STATE.error = ERR_NO_MATCHING;
                break;    /* no next element -- can't get a matching pair */
#endif
            }
            t = p->value;
            x0 = t >> 1;
            winding = which[t & 1];

            /* get the matching transition -- most probably 1 iteration */
            do
            {
                p = nodes + index;
                index = p->next;
                t = p->value;
                x1 = t >> 1;
                winding += which[t & 1];
            }
            while (winding != 0 && index >= 0);

            if (winding != 0)
            {
                STATE.error = ERR_NO_MATCHING;
                break;    /* didn't get a matching pair */
            }

            /* what's the run look like */
            c0 = FS_ROUND(x0) - lo_x;    /* first affected bit */
            c1 = FS_ROUND(x1) - lo_x;    /* 1+last affected bit */
            if (c0 < 0 || c1 > bitmap->width)
            {
                /* should something actually be done in this case?? */
#ifdef FS_DEBUG
                FS_PRINTF(("oops c0=%d c1=%d bitmap->width=%d\n", c0, c1, bitmap->width));
#endif
            }

            /* degenerate runs are dropouts ... fix them? */
            if (c0 == c1)
            {
                if (fix)
                {
                    x0 = (x0 + x1) >> 1;
                    c0 = FS_FLOOR(x0) - lo_x;
                    bits[c0 >> 3] |= fs_mask[c0 & 7];
                }
            }
            else /* set bits from c0 upto but not including c1 */
            {
                FS_BYTE *q;

                c1--;                    /* the last affected bit */
                n = c0 >> 3;
                q = bits + n;            /* ptr first affected byte */
                n = 1 + (c1 >> 3) - n;    /* number of affected bytes */

                if (n > 2)
                {
                    *q++ |= Lmask[c0 & 7];
                    /* unwrap the loop for small n */
                    switch (n - 2)
                    {
                    case 8:
                        *q++ = 0xFF; /* FALLTHROUGH */
                    case 7:
                        *q++ = 0xFF; /* FALLTHROUGH */
                    case 6:
                        *q++ = 0xFF; /* FALLTHROUGH */
                    case 5:
                        *q++ = 0xFF; /* FALLTHROUGH */
                    case 4:
                        *q++ = 0xFF; /* FALLTHROUGH */
                    case 3:
                        *q++ = 0xFF; /* FALLTHROUGH */
                    case 2:
                        *q++ = 0xFF; /* FALLTHROUGH */
                    case 1:
                        *q++ = 0xFF; /* FALLTHROUGH */
                        break;
                    default:
                        SYS_MEMSET(q, 0xFF, n - 2);
                        q += n - 2;
                        break;
                    }
                    *q |= Rmask[c1 & 7];
                }
                else if (n > 1)
                {
                    *q++ |= Lmask[c0 & 7];
                    *q |= Rmask[c1 & 7];
                }
                else /* n==1 */
                    *q |= (Lmask[c0 & 7] & Rmask[c1 & 7]);
            }
        }
        /* advance to next row of bits */
        bits += bitmap->bpl;
    }
}
/*lint +e661 Warning -- Possible access of out-of-bounds pointer         */
/*lint +e662 Warning -- Possible creation of out-of-bounds pointer       */

/****************************************************************/
/* y dropouts -- note: x and y are reversed in tlist */
static FS_VOID y_dropouts(FS_BITMAP *bmap, TLIST *tlist)
{
    int i, winding, row, col;
    FS_FIXED y, y0, y1, t;
    FS_BYTE *bits;
    int which[2] = { -1, 1};
    TNODE *p, *nodes = (TNODE *)(tlist->nodes);

    for (i = 0; i < tlist->num; i++)
    {
        register FS_LONG index;
        index = tlist->indices[i];

        /* process a row of intercepts */
        while (index >= 0)
        {
            /* get next transition */
            p = nodes + index;
            index = p->next;
            if (index < 0)
                break;    /* can't get a matching pair */
            t = p->value;
            y0 = t >> 1;
            winding = which[t & 1];

            /* get the matching transition -- most probably 1 iteration */
            do
            {
                p = nodes + index;
                index = p->next;
                t = p->value;
                y1 = t >> 1;
                winding += which[t & 1];
            }
            while (winding != 0 && index >= 0);

            if (winding != 0)
                break;    /* didn't get a matching pair */

            /* make sure the middle bit of each (thin) run is set */
            if ((y1 - y0) <= FIXED_ONE)
            {
                y = (y0 + y1) >> 1;
                row = bmap->hi_y - FS_FLOOR(y);
                col = (tlist->max - i) - bmap->lo_x;

                /*** this test shouldn't be necessary, BUT ... ***/
                if (row >= 0 && row < bmap->height && col >= 0 && col < bmap->width)
                {
                    bits = bmap->bits + (row * bmap->bpl);
                    bits[col >> 3] |= fs_mask[col & 7];
                }
            }
        }
    }
}

/****************************************************************/
/* turn a scaled outline into a bitmap -- 1 bit per pixel */
FS_BITMAP *make_bitmap(_DS_ FS_OUTLINE *outl)
{
    int i;
    FS_FIXED *x, *y, x0 = 0, y0 = 0;
    FS_BITMAP *bmap;
    TLIST *tlist;
#ifdef FS_STIK
    int is_stik;
    int stik_char;
#endif
    FS_BYTE *type;
    FS_BOOLEAN fix_dropouts = 0;


#ifdef FS_CONTOUR_WINDING_DETECTION
    FS_USHORT contour_number = 0;   /* current contour number */
#endif

    /* ? no outline -- quit */
    if ( outl == 0)
        return 0;

    /* ? can't make new bitmap -- quit */
    bmap = new_bitmap(_PS_ outl);
    if (bmap == 0)
        return 0;

    /* ? space character -- we're done */
    if (outl->num == 0)
    {
        bmap->embedded = false;        /* this bitmap was outline-generated */
        return bmap;
    }

#ifdef FS_STIK
    /* save stik character info. for dropout control */
    stik_char = !(outl->type[0] & OUTLINE_CHAR);

    /* ? is this a stik character */
    is_stik = (STATE.cur_lfnt->fontflags & FONTFLAG_STIK) && stik_char;
    if (is_stik)
    {
        /* ? can we draw it directly */
        if ((STATE.cur_lfnt->fontflags & FONTFLAG_DIRECT_DRAW_ON) &&
            (STATE.cur_sfnt->senv->stroke_width <= 4) &&
            !(STATE.flags & FLAGS_EMBOSSED) &&
            !(STATE.flags & FLAGS_ENGRAVED) &&
            !(STATE.flags & FLAGS_OUTLINED_1PIXEL) &&
            !(STATE.flags & FLAGS_OUTLINED_2PIXEL) &&
            !(STATE.flags & FLAGS_OUTLINED_UNFILLED) &&
            !(STATE.flags & FLAGS_OUTLINED_FILLED)

#ifdef FS_PSEUDO_BOLD
            && ( STATE.cur_sfnt->senv->bold_width == 0)
#endif /* FS_PSEUDO_BOLD */
           )
        {
            /* yes!  we can draw it directly */
            STATE.server->bmap = bmap;
            /* outl passed in will be deleted by caller of make_bitmap */
            draw_stik(_PS_ outl);
            bmap = STATE.server->bmap;    /*** autohint can change STATE.server->bmap !!! */
            STATE.server->bmap = 0;

#ifdef FS_STATS
            if (bmap) made_bmap++;
#endif
            if (bmap)
                bmap->embedded = false;        /* this bitmap was outline-generated */
            return bmap;
        }
        else    /* no ... expand to outline and treat normally */
        {
#ifdef FS_PSEUDO_BOLD
            FS_OUTLINE *temp = (FS_OUTLINE *)NULL;
#endif
            FS_LONG npoints, ntypes;

            /* outl passed in will be deleted by caller of make_bitmap */
            /* outl returned will be deleted by "if (is_stik)" below */
            outl = bitmap_expand_stik(_PS_ outl, &ntypes, &npoints);
            if (!outl)
            {
                if (bmap)
                    bmap->embedded = false;
                return bmap;
            }
#ifdef FS_PSEUDO_BOLD
            if (STATE.bold_pct || STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->bold_pct)
            {
                /* embolden outline for FLAGS_ADD_WEIGHT off                   */
                /*    and smaller than 26 ppem NEW_AA font with bold_pct not 0 */
                if ( ((STATE.flags & FLAGS_OUTLINEBOLD)) ||
                     ( !(STATE.flags & FLAGS_ADD_WEIGHT) &&
                     ( !( STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON) || STATE.lpm > 26)) )
                {
                    FS_BITMAP *bold_bmap;
                    temp = embolden_outline(_PS_ outl, &ntypes, &npoints);
                    if (!temp)
                    {
                        FSS_free_char(_PS_ outl);
                        if (bmap)
                            bmap->embedded = false;
                        return bmap;
                    }
                    if (temp != outl)
                    {
                        FSS_free_char(_PS_ outl);
                        outl = temp;
                    }
                    bold_bmap = new_bitmap(_PS_ outl);
                    if (bold_bmap == 0)
                    {
                        FSS_free_char(_PS_ bmap);
                        return 0;
                    }
                    FSS_free_char(_PS_ bmap);
                    bmap = bold_bmap;
                }
            }
#endif /* pseudo-bold */
        }
    }

    if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK) && outl->num && !stik_char) /* 05/17/04 qw */
    {
#ifdef FS_CONTOUR_WINDING_DETECTION
        /* fix dropouts for an outline char in a stik font */
        /*   unless we are just checking contour winding */
        if (STATE.lpm < 64 && (!(STATE.flags & FLAGS_CHECK_CONTOUR_WINDING_ON)))
#else
        /* fix dropouts for an outline char in a stik font */
        if (STATE.lpm < 64)
#endif
            STATE.flags |= FLAGS_FIX_DROPOUTS;
    }
#endif /* FS_STIK */


    /* allocate transition list */
    STATE.server->tlist = tlist = new_tlist(_PS_ (TLIST *)(STATE.server->tlist), bmap->hi_y, bmap->height);
    if (tlist == 0)
    {
        FSS_free_char(_PS_ bmap);/* 03-18-04 FA fatal error got to free bmap */
        bmap = 0; /* and set bmap to NULL so trim_bitmap returns gracefully */
        STATE.error = ERR_MALLOC_FAIL; /* mem allocation in call to new_tlist failed */
#ifdef FS_STIK
        if (is_stik)
            FSS_free_char(_PS_ outl);
#endif
        return bmap;
    }
    STATE.server->tlist->ref_count++;

    /* set up for simple rasterization OR dropout control */
    STATE.server->line = line_simple;

    if (STATE.flags & FLAGS_FIX_DROPOUTS)
    {
        STATE.server->drop = new_tlist(_PS_ (TLIST *)(STATE.server->drop), (FS_SHORT)(bmap->width + bmap->lo_x - 1), bmap->width);
        if (STATE.server->drop == 0)
            STATE.error = SUCCESS;    /* not really fatal - just no y-dropout control */
        else
        {
            STATE.server->line = line_dropout;
            fix_dropouts = 1;
            STATE.server->drop->ref_count++;
        }
    }
#ifdef FS_CONTOUR_WINDING_DETECTION
    /* If we're checking contours,  */
    if (STATE.flags & FLAGS_CHECK_CONTOUR_WINDING_ON)
    {
        FS_LONG *nextLoopTX;
        /* allocate an array to remember the "just past end" TNODE indices */
        /* of each loop */
#ifdef FS_MEM_DBG
        STATE.memdbgid = "STATE.server->nextLoopTX";
#endif
        nextLoopTX = (FS_LONG *) FSS_calloc(_PS_ outl->nc * sizeof(FS_LONG));
        if (!nextLoopTX )
        {
            /* clean up and bail */
            FSS_free_char(_PS_ bmap); /* just needed it for bmap->height */
            bmap = 0;
#ifdef FS_STIK
            if (is_stik)
                FSS_free_char(_PS_ outl);
#endif
            STATE.server->tlist->ref_count--;
            if (STATE.server->drop != 0)
                STATE.server->drop->ref_count--;
            return bmap;
        }
        STATE.server->nextLoopTX = nextLoopTX;
    }
#endif

    /* generate some pixels */
    x = (FS_FIXED *)(outl->x);
    y = (FS_FIXED *)(outl->y);
    type = (FS_BYTE *)(outl->type);

    for (i = 0; i < outl->num; i++)
    {
        switch (type[i] & 0x7f)
        {
        case FS_MOVETO:
#ifdef FS_CONTOUR_WINDING_DETECTION
            /* If we're checking contours,  */
            if (STATE.flags & FLAGS_CHECK_CONTOUR_WINDING_ON)
                if (i) /* don't save first moveto when i==0 */
                    /* add to nextLoopTX */
                    STATE.server->nextLoopTX[contour_number++] = STATE.server->tlist->next_index;
#endif

            x++;
            y++;
            if (i != 0)
            {
                TLIST *cur_tlist = (TLIST *)(STATE.server->tlist);
                int c0 = FS_ROUND((cur_tlist->hph_prevz >> 1));  /* first affected bit */
                int c1 = FS_ROUND((cur_tlist->hph_loopstartz >> 1));  /* 1+last affected bit */
                if ( (cur_tlist->hph_prevz & 1) != (cur_tlist->hph_loopstartz & 1) &&
                        c0 == c1 && cur_tlist->hph_prevy == cur_tlist->hph_loopstarty)
                {
                    remove_tran(cur_tlist->hph_prevz, cur_tlist->hph_prevy, cur_tlist);
                    remove_tran(cur_tlist->hph_loopstartz, cur_tlist->hph_loopstarty, cur_tlist);
                }

                cur_tlist = (TLIST *)(STATE.server->drop);
                if (fix_dropouts)
                {
                    c0 = FS_ROUND((cur_tlist->hph_prevz >> 1));  /* first affected bit */
                    c1 = FS_ROUND((cur_tlist->hph_loopstartz >> 1));  /* 1+last affected bit */
                    if ( (cur_tlist->hph_prevz & 1) != (cur_tlist->hph_loopstartz & 1) &&
                            c0 == c1 && cur_tlist->hph_prevy == cur_tlist->hph_loopstarty)
                    {
                        remove_tran(cur_tlist->hph_prevz, cur_tlist->hph_prevy, cur_tlist);
                        remove_tran(cur_tlist->hph_loopstartz, cur_tlist->hph_loopstarty, cur_tlist);
                    }
                }
            }

            if (fix_dropouts)
            {
                STATE.server->drop->hph_flag = 0;
            }
            STATE.server->tlist->hph_flag = 0;
            break;

        case FS_LINETO:
            (*STATE.server->line)(_PS_ x0, y0, x[0], y[0]);
            x++;
            y++;
            break;
        case FS_QUADTO:
            FS_quad(_PS_ x0, y0, x[0], y[0], x[1], y[1]);
            x += 2;
            y += 2;
            break;
        case FS_CUBETO:
            FS_cube(_PS_ x0, y0, x[0], y[0], x[1], y[1], x[2], y[2]);
            x += 3;
            y += 3;
            break;
        default:
            break;
        }
        x0 = x[-1];
        y0 = y[-1];

        if (STATE.error)
        {
            FSS_free_char(_PS_ bmap);
            bmap = 0;
#ifdef FS_STIK
            if (is_stik)
                FSS_free_char(_PS_ outl);
#endif
            STATE.server->tlist->ref_count--;
            if (STATE.server->drop != 0)
                STATE.server->drop->ref_count--;
            return bmap;
        }
    }


    {
        /* check last loop */
        TLIST *cur_tlist = (TLIST *)(STATE.server->tlist);
        int c0 = FS_ROUND((cur_tlist->hph_prevz >> 1));  /* first affected bit */
        int c1 = FS_ROUND((cur_tlist->hph_loopstartz >> 1));  /* 1+last affected bit */
        if ( (cur_tlist->hph_prevz & 1) != (cur_tlist->hph_loopstartz & 1) &&
             c0 == c1 && cur_tlist->hph_prevy == cur_tlist->hph_loopstarty )
        {
            remove_tran(cur_tlist->hph_prevz, cur_tlist->hph_prevy, cur_tlist);
            remove_tran(cur_tlist->hph_loopstartz, cur_tlist->hph_loopstarty, cur_tlist);
        }
        cur_tlist->ref_count--;
        cur_tlist = (TLIST *)(STATE.server->drop);
        if (fix_dropouts)
        {
            c0 = FS_ROUND((cur_tlist->hph_prevz >> 1));  /* first affected bit */
            c1 = FS_ROUND((cur_tlist->hph_loopstartz >> 1));  /* 1+last affected bit */
            if ( (cur_tlist->hph_prevz & 1) != (cur_tlist->hph_loopstartz & 1) &&
                 c0 == c1 && cur_tlist->hph_prevy == cur_tlist->hph_loopstarty )
            {
                remove_tran(cur_tlist->hph_prevz, cur_tlist->hph_prevy, cur_tlist);
                remove_tran(cur_tlist->hph_loopstartz, cur_tlist->hph_loopstarty, cur_tlist);
            }
            cur_tlist->ref_count--;
        }
    }
#ifdef FS_CONTOUR_WINDING_DETECTION
    /* If we're checking contours,  */
    if (STATE.flags & FLAGS_CHECK_CONTOUR_WINDING_ON)
    {
        /* fill in last TNODE endpoint */
        STATE.server->nextLoopTX[contour_number] = STATE.server->tlist->next_index;
        FSS_free_char(_PS_ bmap); /* just needed it for bmap->height */
        /* return and let check_contour_winding() analyze the tlist */
        return 0;
    }
#endif

    /* set the runs, discard the tlist */
    nzw_runs(_PS_ bmap, STATE.server->tlist);

    if (fix_dropouts)
    {
        y_dropouts(bmap, STATE.server->drop);
    }
#ifdef FS_STATS
    if (bmap) made_bmap++;
#endif

#ifdef FS_STIK
    if (is_stik)
        FSS_free_char(_PS_ outl);
#endif

#ifdef FS_PSEUDO_BOLD /* need to be done before trim */
    if ( (STATE.bold_pct || STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->bold_pct))
    {
        FS_BOOLEAN pixel_bold;
        pixel_bold = ( (STATE.flags & FLAGS_ADD_WEIGHT) ||
                       ((STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON) &&
                        (STATE.lpm <= 26) && !(STATE.flags & FLAGS_OUTLINEBOLD)));
        if ( pixel_bold )
        {
            if (bmap)
                pixelbold_bitmap(_PS_ bmap);
        }
    }
#endif

    if (bmap)
        bmap->embedded = false;        /* this bitmap was outline-generated */
    return bmap;
}

#endif /* FS_BITMAPS */

#if defined(FS_BITMAPS) || defined(FS_EMBEDDED_BITMAP)
/****************************************************************/
/* remove leading/trailing blank rows/columns from bitmap */
FS_BITMAP *trim_bitmap(_DS_ FS_BITMAP *bmap)
{
#ifdef FS_TRIM_BITMAP
    FS_BYTE *bp;
    int i, j;
    FS_LONG size;
    int bpl;
    FS_SHORT height, width, l_shift, r_shift;
    FS_BYTE mask;

    if (bmap == 0)
        return 0;

    /* leading empty rows */
    height = bmap->height;
    for (; bmap->height;)
    {
        bp = bmap->bits + (height - bmap->height) * bmap->bpl;
        for (i = 0; i < bmap->bpl; i++)
            if (bp[i])
                break;

        if (i != bmap->bpl)
            break;

        bmap->height--;
        bmap->hi_y--;
    }

    height -= bmap->height;
    /* trailing empty rows */
    for (; bmap->height;)
    {
        bp = bmap->bits + bmap->bpl * (bmap->height + height - 1);
        for (i = 0; i < bmap->bpl; i++)
            if (bp[i])
                break;

        if (i != bmap->bpl)
            break;

        bmap->height--;
    }

    /* leading blank columns */
    width = bmap->width;
    mask = 0x80;
    for (; bmap->width;)
    {
        bp = bmap->bits + height * bmap->bpl;
        for (i = 0; i < bmap->height; i++)
        {
            if ( *bp & mask )
                break;
            bp += bmap->bpl;
        }

        if (i != bmap->height)
            break;

        bmap->lo_x++;
        bmap->width--;
        mask = mask >> 1;
        if (mask == 0 )
        {
            bp = bmap->bits + height * bmap->bpl;
            for (i = 0; i < bmap->height; i++)
            {
                for (j = 1; j < bmap->bpl; j++)
                    bp[j - 1] = (bp[j - 1] << 8) | bp[j];
                bp[bmap->bpl - 1] = bp[bmap->bpl - 1] << 8;
                bp += bmap->bpl;
            }
            mask = 0x80;
        }
    }
    width -= bmap->width;
    l_shift = width % 8;
    r_shift = 8 - l_shift;
    if (l_shift)
    {
        bp = bmap->bits + height * bmap->bpl;
        for (i = 0; i < bmap->height; i++)
        {
            for (j = 1; j < bmap->bpl; j++)
                bp[j - 1] = (bp[j - 1] << l_shift) | (bp[j] >> r_shift );
            bp[bmap->bpl - 1] = bp[bmap->bpl - 1] << l_shift;
            bp += bmap->bpl;
        }
    }

    /* trailing blank columns */
    for (; bmap->width;)
    {
        FS_BYTE last_bit;
        FS_SHORT last_byte;

        last_bit = fs_mask[(bmap->width - 1) & 7];
        last_byte = (bmap->width +  - 1) >> 3;
        bp = bmap->bits + last_byte + height * bmap->bpl;

        for (i = 0; i < bmap->height; i++)
        {
            if (*bp & last_bit)
                break;
            bp += bmap->bpl;
        }

        if (i != bmap->height)
            break;

        /* note the change in width */
        bmap->width--;
    }

    /* now bmap is trimmed and valid -- remember bpl != (7 + width)/8 is OK,
    * but possibly too large if we've deleted rows or lots of columns, so
    * make a nice copy of proper allocation size -- and no trailing unused bits.
    */
    size = offsetof(FS_BITMAP, bits);
    bpl = (7 + bmap->width) / 8;
    size += bpl * bmap->height;

    if (size != bmap->size)
    {
        FS_BITMAP *dst;
        FS_BYTE *sp, *dp;

#ifdef FS_MEM_DBG
        STATE.memdbgid = "FS_BITMAP";
#endif
        dst = (FS_BITMAP *)FSS_calloc(_PS_ size);

        if (dst == 0)
        {
            /* it's not a FATAL error */
            STATE.error = SUCCESS;
            return bmap;
        }
        dst->size = size;
        dst->lo_x = bmap->lo_x;
        dst->hi_y = bmap->hi_y;
        dst->width = bmap->width;
        dst->height = bmap->height;
        dst->bpl = (FS_SHORT)bpl;
        dst->i_dx = bmap->i_dx;
        dst->i_dy = bmap->i_dy;
        dst->dx = bmap->dx;
        dst->dy = bmap->dy;
        dst->type = FS_MAP_BITMAP;
        dst->bitsPerPixel = bmap->bitsPerPixel;

        /* copy the relevant bits */
        sp = bmap->bits + height * bmap->bpl;
        dp = dst->bits;
        if (bmap->bpl == dst->bpl)
            SYS_MEMCPY(dp, sp, dst->bpl * dst->height);
        else
        {
            /* copy partial rows at a time */
            for (i = 0; i < dst->height; i++)
            {
                SYS_MEMCPY(dp, sp, dst->bpl);
                dp += dst->bpl;
                sp += bmap->bpl;
            }
        }

        FSS_free_char(_PS_ bmap);
        bmap = dst;
    }
#endif /* FS_TRIM_BITMAP */
    STATE.error = SUCCESS;
    return bmap;
}

#ifdef FS_BITMAPS
static FS_BITMAP *
construct_bitmap(_DS_ FS_FIXED *scale, FS_BOOLEAN vanilla, FS_OUTLINE *outl)
{
    FS_BITMAP *bmap;

#ifdef FS_CONTOUR_WINDING_DETECTION
    {
        FS_ULONG flags = STATE.flags;
        STATE.flags &= FLAGS_CHECK_CONTOUR_WINDING_OFF;
        bmap = make_bitmap(_PS_ outl);
        STATE.flags = flags;
    }
#else
    bmap = make_bitmap(_PS_ outl);
#endif
    FSS_free_char(_PS_ outl);

    if (bmap == 0)  /* Coverity does not like "if (STATE.error)" */
    {
        return 0;
    }
    if (STATE.flags & FLAGS_EMBOSSED)
    {
        bmap = emboss_bitmap(bmap);
    }
    else if (STATE.flags & FLAGS_ENGRAVED)
    {
        bmap = engrave_bitmap(bmap);
    }
    else if (STATE.flags & FLAGS_OUTLINED_1PIXEL)
    {
        bmap = outline_bitmap(_PS_ bmap, 1, 0);
    }
    else if (STATE.flags & FLAGS_OUTLINED_2PIXEL)
    {
        bmap = outline_bitmap(_PS_ bmap, 2, 0);
    }
    else if ( STATE.flags & (FLAGS_OUTLINED_UNFILLED | FLAGS_OUTLINED_FILLED | FLAGS_OUTLINED_SOFT) )
    {
        bmap = outline_bitmap(_PS_ bmap, STATE.outline_width,
                              (FS_USHORT)!(STATE.flags & FLAGS_OUTLINED_UNFILLED));
    }

    if ((bmap == 0) || (STATE.error && STATE.error != ERR_NO_MATCHING))
    {
        if (bmap) FSS_free_char(_PS_ bmap);
        return 0;
    }
    /* Install method to handle the number line issue in rotational */
    /* modes; 90 and 180 rotations require adjustments, in order to */
    /* align correctly with respect to baseline. We need to disable */
    /* embedded bitmap data from this case.                         */
    if (!vanilla && !bmap->embedded)
    {
        /* fix rotation cases. */

        /* 180 degree rotation, or...        */
        /*   0 degree rotation, and y-mirror */
        if (scale[3] < 0)
            bmap->hi_y++;

        /* 270 degree rotation, or...        */
        /* 090 degree rotation, and x-mirror */
        if (scale[2] < 0)
            bmap->hi_y++;

        /* 090 degree rotation, or...        */
        /* 270 degree rotation, and y-mirror */
        if (scale[1] < 0)
            bmap->lo_x++;

        /* 180 degree rotation, or...        */
        /*   0 degree rotation, and x-mirror */
        if (scale[0] < 0)
            bmap->lo_x++;
    }
    bmap = trim_bitmap(_PS_ bmap);
    return bmap;
}
#endif /* FS_BITMAPS */

/****************************************************************/
FS_BITMAP *get_bitmap(_DS_ FS_ULONG id, FS_USHORT index)
{
    FS_BITMAP *bmap = 0;
    SFNT *sfnt;
    LFNT *lfnt;
    FS_ULONG flags = STATE.flags;

    STATE.error = SUCCESS;

#if defined(FS_BITMAPS) && defined(FS_EXTERNAL_OUTLINE)
    if (STATE.user_outline)
    {
        FS_OUTLINE *outl;

        outl = copy_outline(_PS_ STATE.user_outline,
                            0 /* source is in server memory */);
        if (!outl) return 0;

        bmap = construct_bitmap(_PS_ STATE.user_outline_scale,
                                STATE.user_outline_scale_vanilla, outl);
        if (!bmap) return 0;

        return bmap;
    }
#endif

    lfnt = STATE.cur_lfnt;
    sfnt = STATE.cur_sfnt;
    if (lfnt == 0 || sfnt == 0)
        return 0;

    STATE.flags &= ~FLAGS_GRAYSCALE; /* no grayscale autohinting */

#ifdef FS_CACHE_BITMAPS
    bmap = find_bitmap_in_cache(_PS_ index); /*lint !e838 Suppress 'Previously assigned value has not been used' */
    if (bmap)
    {
        STATE.flags = flags;
        return bmap;
    }
#endif

#ifdef FS_EMBEDDED_BITMAP
    if ((lfnt->fnt_type == TTF_TYPE) || (lfnt->fnt_type == TTC_TYPE))
    {
        bmap = get_embedded_bitmap(_PS_ sfnt, index);
        if (STATE.error)
        {
            STATE.flags = flags;
            return 0;
        }
        /* only apply special effects when FS_BITMAPS defined */
#ifdef FS_BITMAPS
        if (bmap)
        {
            /* apply any requested effects ... and cache the result */
            if (STATE.flags & (FLAGS_EMBOSSED | FLAGS_ENGRAVED | FLAGS_OUTLINED_1PIXEL |
                               FLAGS_OUTLINED_2PIXEL | FLAGS_OUTLINED_UNFILLED | FLAGS_OUTLINED_FILLED |
                               FLAGS_OUTLINED_SOFT))
            {
                if (STATE.flags & FLAGS_EMBOSSED)
                    bmap = emboss_bitmap(bmap);
                else if (STATE.flags & FLAGS_ENGRAVED)
                    bmap = engrave_bitmap(bmap);
                else if (STATE.flags & FLAGS_OUTLINED_1PIXEL)
                {
                    bmap = outline_bitmap(_PS_ bmap, 1, 0);
                    bmap = trim_bitmap(_PS_ bmap);
                }
                else if (STATE.flags & FLAGS_OUTLINED_2PIXEL)
                {
                    bmap = outline_bitmap(_PS_ bmap, 2, 0);
                    bmap = trim_bitmap(_PS_ bmap);
                }
                else if (STATE.flags & (FLAGS_OUTLINED_UNFILLED | FLAGS_OUTLINED_FILLED | FLAGS_OUTLINED_SOFT))
                {
                    bmap = outline_bitmap(_PS_ bmap, STATE.outline_width,
                                          (FS_USHORT)!(STATE.flags & FLAGS_OUTLINED_UNFILLED) );
                    bmap = trim_bitmap(_PS_ bmap);
                }
#ifdef FS_CACHE_BITMAPS
                save_bitmap_to_cache(_PS_ index, bmap);
#endif
            }
            else
            {
#ifdef FS_PSEUDO_BOLD
                {
                    FS_SHORT bw = sfnt->senv->bold_width;
                    if (bw)
                    {
                        bmap = pixelbold_embedded_bitmap(_PS_ bmap);
#ifdef FS_CACHE_BITMAPS
                        save_bitmap_to_cache(_PS_ index, bmap);
#endif
                    }
                }
#endif
            }
        }
#endif    /* FS_BITMAPS */
    }
#endif  /* FS_EMBEDDED_BITMAP */


#ifdef FS_BITMAPS

    if (!bmap)
    {
        FS_OUTLINE *outl;

        outl = find_or_make_outline(_PS_ lfnt, sfnt, id, index);
        if (!outl)
        {
            STATE.flags = flags;
            return 0;
        }

        bmap = construct_bitmap(_PS_ sfnt->user_scale,
                                sfnt->senv->vanilla, outl);
        if (!bmap)
        {
            STATE.flags = flags;
            return 0;
        }

        /* apply any vertical shift adjustment for component font */
        if (sfnt->vertical_shift)
        {
            SENV *senv = (SENV *)(sfnt->senv);

            if (senv) /* should be true always at this point */
            {
                if (senv->vanilla)
                {
                    bmap->hi_y += sfnt->vertical_shift;
                }
                else
                {
                    FIXED_VECTOR p;
                    p.x = bmap->dx;
                    p.y = bmap->dy;
                    fixed_norm(&p);  /* unit tangent vector */
                    /* adjust bitmap placement by scaled normal vector */
                    bmap->lo_x += (FS_SHORT)FixMul(-p.y, sfnt->vertical_shift);
                    bmap->hi_y += (FS_SHORT)FixMul( p.x, sfnt->vertical_shift);
                }
            }
        }

#ifdef FS_CACHE_BITMAPS
        save_bitmap_to_cache(_PS_ index, bmap);
#endif
    }
    bmap->bitsPerPixel = 1;
    bmap->type = FS_MAP_BITMAP;
#else  /* FS_BITMAPS */
    id = id; /* fix compiler warning when FS_BITMAPS not defined */
#endif /* FS_BITMAPS */
    STATE.flags = flags;
    return bmap;
}
#endif /* defined(FS_BITMAPS) || defined(FS_EMBEDDED_BITMAP) */
/****************************************************************/
/* display the bitmap in ASCII */
#ifdef FS_DUMP
FS_VOID dump_bitmap(FS_BITMAP *bitmap)
{
    int i, j, index;
    FS_BYTE bit, *p;
    int lo_x ;
    int hi_y;
    int width;
    int height;
    int bpl;
    int i_dx;
    int i_dy;
    double dx;
    double dy;

    if (!bitmap)
        return;

    lo_x = bitmap->lo_x;
    hi_y = bitmap->hi_y;
    width = bitmap->width;
    height = bitmap->height;
    bpl = bitmap->bpl;
    i_dx = bitmap->i_dx;
    i_dy = bitmap->i_dy;
    dx = bitmap->dx / 65536.0;
    dy = bitmap->dy / 65536.0;

    FS_PRINTF(("lo_x=%d hi_y=%d width=%d height=%d bpl=%d\n", lo_x, hi_y, width, height, bpl));
    FS_PRINTF(("i_dx=%d i_dy=%d dx=%12.5f dy=%12.5f\n", i_dx, i_dy, dx, dy));

    /* 3 digit x coords across the top */
    FS_PRINTF(("    "));

    for (i = 0; i < width; i++)
        FS_PRINTF(("%d", ABS(lo_x + i) / 100));

    FS_PRINTF(("\n    "));
    for (i = 0; i < width; i++)
        FS_PRINTF(("%d", (ABS(lo_x + i) % 100) / 10));

    FS_PRINTF(("\n    "));
    for (i = 0; i < width; i++)
        FS_PRINTF(("%d", ABS(lo_x + i) % 10));
    FS_PRINTF(("\n"));

    /* now the bitmap */
    p = bitmap->bits;
    for (i = 0; i < height; i++)
    {
        /* 3 digit y coords */
        FS_PRINTF(("%3d ", hi_y - i));
        index = 0;
        bit = 0x80;
        for (j = 0; j < width; j++)
        {
            if (p[index] & bit)
            {
                FS_PRINTF(("%c", '*'));
            }
            else
            {
                FS_PRINTF(("%c", '-'));
            }

            bit >>= 1;
            if (bit == 0)
            {
                index++;
                bit = 0x80;
            }
        }
        FS_PRINTF(("%c", ' '));
        for (j = 0; j < bpl; j++)
            FS_PRINTF(("%02x ", p[j]));
        FS_PRINTF(("\n"));
        p += bpl;
    }
    FS_PRINTF(("\n"));
}
#endif    /* FS_DUMP */
